<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\User\Sales;
use App\Models\User\SalesPayment;
use App\Models\User\Line;
use App\Models\User\Member;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SalesPaymentController extends Controller
{
    /* =======================
       INDEX
    ======================== */
    public function index()
    {
        return view('user.sales_payments.index');
    }

    /* =======================
       LINE AUTOCOMPLETE
    ======================== */
    public function searchLines(Request $request)
    {
        $q = trim($request->q);

        if ($q === '') {
            return response()->json([]);
        }

        $userId = session('user_id');

        return Line::where('user_id', $userId)
        ->where('type','Customer')
            ->where('line_name', 'like', "{$q}%")
            ->orderBy('line_name')
            ->limit(15)
            ->get([
                'line_id as id',
                'line_name as name'
            ]);
    }

    /* =======================
       CUSTOMER AUTOCOMPLETE
    ======================== */
    public function searchCustomer(Request $request)
    {
        $q = trim($request->query('q', ''));

        if ($q === '') {
            return response()->json([]);
        }

        $userId = session('user_id');

        return Member::where('user_id', $userId)
            ->where('type', 'Customer')
            ->where('name', 'like', "{$q}%")
            ->limit(15)
            ->get()
            ->map(function ($r) {
                return [
                    'id'      => $r->id,
                    'name'    => $r->name,
                    'phone'   => $r->phone ?? '',
                    'address' => $r->address ?? '',
                ];
            });
    }

    /* =======================
       BALANCE HELPER
    ======================== */
    private function getCustomerBalanceAmount($userId, $customerId, $lineId, $date)
    {
        // Total sales till date
        $totalSales = Sales::where('user_id', $userId)
            ->where('customer_id', $customerId)
            ->where('line_id', $lineId)
            ->whereDate('date', '<=', $date)
            ->sum('total_amount');

        // Total received till date
        $totalReceived = Sales::where('user_id', $userId)
            ->where('customer_id', $customerId)
            ->where('line_id', $lineId)
            ->whereDate('date', '<=', $date)
            ->sum('paid_amount');

        return max(0, $totalSales - $totalReceived);
    }

    /* =======================
       FETCH BALANCE (AJAX)
    ======================== */
    public function getBalance(Request $request)
    {
        $lineId     = $request->line_id;
        $customerId = $request->customer_id;
        $date       = $request->payment_date;

        if (!$lineId || !$customerId || !$date) {
            return response()->json([
                'total' => 0,
                'paid' => 0,
                'balance' => 0
            ]);
        }

        $userId = session('user_id');

        $totalSales = Sales::where('user_id', $userId)
            ->where('customer_id', $customerId)
            ->where('line_id', $lineId)
            ->whereDate('date', '<=', $date)
            ->sum('total_amount');

        $totalReceived = Sales::where('user_id', $userId)
            ->where('customer_id', $customerId)
            ->where('line_id', $lineId)
            ->whereDate('date', '<=', $date)
            ->sum('paid_amount');

        return response()->json([
            'total'   => $totalSales,
            'paid'    => $totalReceived,
            'balance' => $totalSales - $totalReceived
        ]);
    }

    /* =======================
       STORE SALES PAYMENT
    ======================== */
    public function store(Request $request)
    {
        $request->validate([
            'payment_date'  => 'required|date',
            'line_id'       => 'required|integer',
            'line_name'     => 'required|string',
            'customer_id'   => 'required|integer',
            'customer_name' => 'required|string',
            'paid_amount'   => 'required|numeric|min:1',
            'description'   => 'nullable|string',
        ]);

        $userId = session('user_id');

        // 1️⃣ Get balance till date
        $balance = $this->getCustomerBalanceAmount(
            $userId,
            $request->customer_id,
            $request->line_id,
            $request->payment_date
        );

        // 2️⃣ Prevent over-receipt
        if ($request->paid_amount > $balance) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Received amount exceeds customer balance'
            ], 422);
        }

        DB::transaction(function () use ($request, $userId) {

            // 3️⃣ Save sales payment
            SalesPayment::create([
                'user_id'        => $userId,
                'payment_date'   => $request->payment_date,
                'line_id'        => $request->line_id,
                'line_name'      => $request->line_name,
                'customer_id'    => $request->customer_id,
                'customer_name'  => $request->customer_name,
                'description'    => $request->description,
                'paid_amount'    => $request->paid_amount,
            ]);

            // 4️⃣ Apply payment to sales (FIFO)
            $remaining = $request->paid_amount;

            $sales = Sales::where('user_id', $userId)
                ->where('customer_id', $request->customer_id)
                ->where('line_id', $request->line_id)
                ->whereDate('date', '<=', $request->payment_date)
                ->whereColumn('paid_amount', '<', 'total_amount')
                ->orderBy('date')
                ->orderBy('id')
                ->lockForUpdate()
                ->get();

            foreach ($sales as $sale) {

                if ($remaining <= 0) break;

                $due = $sale->total_amount - $sale->paid_amount;

                if ($remaining >= $due) {
                    $sale->paid_amount += $due;
                    $remaining -= $due;
                } else {
                    $sale->paid_amount += $remaining;
                    $remaining = 0;
                }

                $sale->save();
            }
        });

        return response()->json([
            'status'  => 'success',
            'message' => 'Sales payment saved and sales updated successfully'
        ]);
    }
}
