<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\User\Member;
use Illuminate\Http\Request;

class MemberController extends Controller
{
    // -----------------------------------------------------
    // LIST MEMBERS PAGE (create view contains list also)
    // -----------------------------------------------------
    public function index()
    {
        return redirect()->route('user.members.create');
    }

    // -----------------------------------------------------
    // SHOW MAIN PAGE (ACTIVE + DELETED)
    // -----------------------------------------------------
    public function create()
    {
        $userId = session('user_id');

        // active
        $customers = Member::where('user_id', $userId)
            ->where('type', 'Customer')
            ->get();

        $suppliers = Member::where('user_id', $userId)
            ->where('type', 'Supplier')
            ->get();

        // deleted
        $customersDeleted = Member::onlyTrashed()
            ->where('user_id', $userId)
            ->where('type', 'Customer')
            ->get();

        $suppliersDeleted = Member::onlyTrashed()
            ->where('user_id', $userId)
            ->where('type', 'Supplier')
            ->get();

        return view(
            'user.members.create',
            compact('customers', 'suppliers', 'customersDeleted', 'suppliersDeleted')
        );
    }

    // -----------------------------------------------------
    // STORE MEMBER
    // -----------------------------------------------------
    public function store(Request $request)
    {
        $userId = session('user_id');

        $request->validate([
            'name'     => 'required|string|max:255',
            'phone'    => 'required|string|max:20',
            'address'  => 'required|string|max:255',
            'type'     => 'required|in:Customer,Supplier',
        ]);

        $member = Member::create([
            'user_id'   => $userId,
            'name'      => $request->name,
            'phone'     => $request->phone,
            'address'   => $request->address,
            'type'      => $request->type,
        ]);

        if ($request->ajax()) {
            return response()->json([
                'status' => 'ok',
                'member' => $member
            ]);
        }

        return redirect()->back()->with('success', 'Member added successfully');
    }

    // -----------------------------------------------------
    // EDIT PAGE
    // -----------------------------------------------------
 public function edit($id)
{
    $member = Member::findOrFail($id);
        $userId = session('user_id');

        if ($member->user_id != $userId) {
            abort(403, 'Unauthorized');
        }

        return view('user.members.edit', compact('member'));
    }

    // -----------------------------------------------------
    // UPDATE MEMBER
    // -----------------------------------------------------
 public function update(Request $request, Member $member)
{
    if ($request->ajax()) {
        if ($member->user_id != session('user_id')) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $member->update($request->only('name', 'phone', 'address'));

        return response()->json(['member' => $member], 200);
    }

    // non-ajax fallback
    return redirect()->route('user.members.create')
        ->with('success', 'Member updated successfully.');
}


    // -----------------------------------------------------
    // SOFT DELETE MEMBER
    // -----------------------------------------------------
    public function destroy(Request $request, Member $member)
    {
        $userId = session('user_id');

        if ($member->user_id != $userId) {
            abort(403, 'Unauthorized');
        }

        $member->delete();

        if ($request->ajax()) {
            return response()->json([
                'status' => 'success',
                'message' => 'Member deleted successfully',
            ]);
        }

        return redirect()->route('user.members.create')
            ->with('success', 'Member deleted successfully.');
    }

    // -----------------------------------------------------
    // RESTORE DELETED MEMBER
    // -----------------------------------------------------
    public function restore($id)
    {
        $userId = session('user_id');

        $member = Member::onlyTrashed()
            ->where('id', $id)
            ->where('user_id', $userId)
            ->firstOrFail();

        $member->restore();

        return response()->json(['status' => 'ok', 'member' => $member]);
    }
}
