<?php
namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User\Expense;
use App\Models\User\ExpenseCategory;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;

class ExpenseController extends Controller
{
    // show UI
    public function index()
    {
        
        $userId = session('user_id'); // or auth()->id()
        // global categories (user_id null) + user's categories
        $categories = ExpenseCategory::whereNull('user_id')
            ->orWhere('user_id', $userId)
            ->orderByRaw("CASE WHEN user_id IS NULL THEN 0 ELSE 1 END, name")
            ->get();
   
        return view('user.expenses.index', compact('categories'));
    }

    // AJAX store a single expense
    public function store(Request $request)
    {
        $userId = session('user_id');

        $data = $request->validate([
            'category_id' => 'nullable|exists:expense_categories,id',
            'new_category' => 'nullable|string|max:120',
            'amount' => 'required|numeric|min:0.01',
            'title' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:2000',
            'date' => 'nullable|date',
        ]);

        // If new_category provided create it for the user
        if (!empty($data['new_category'])) {
            $cat = ExpenseCategory::create([
                'user_id' => $userId,
                'name' => $data['new_category'],
                'slug' => Str::slug($data['new_category']),
            ]);
            $categoryId = $cat->id;
        } else {
            $categoryId = $data['category_id'] ?? null;
        }

        $expense = Expense::create([
            'user_id' => $userId,
            'category_id' => $categoryId,
            'title' => $data['title'] ?? null,
            'description' => $data['description'] ?? null,
            'amount' => $data['amount'],
            'date' => $data['date'] ?? now()->toDateString(),
        ]);

        return response()->json([
            'success' => true,
            'expense' => $expense->load('category'),
            'message' => 'Saved'
        ]);
    }

    // create category (AJAX) - optional endpoint
    public function createCategory(Request $request)
    {
        $userId = session('user_id');
        $data = $request->validate(['name'=>'required|string|max:120']);
        $cat = ExpenseCategory::create(['user_id'=>$userId, 'name'=>$data['name'], 'slug'=>Str::slug($data['name'])]);
        return response()->json(['success'=>true,'category'=>$cat]);
    }

    // return today's total for the logged user
    public function todayTotal(Request $request)
    {
        $userId = session('user_id');
        $today = now()->toDateString();
        $total = Expense::where('user_id', $userId)->whereDate('date', $today)->sum('amount');
        return response()->json(['total' => number_format($total,2)]);
    }

    // search / list (for panel below)
    public function search(Request $request)
    {
        $userId = session('user_id');
        $query = Expense::with('category')->where('user_id', $userId);

        if ($q = $request->query('q')) {
            $query->where(function($qr) use ($q) {
                $qr->where('title', 'like', "%$q%")
                   ->orWhere('description', 'like', "%$q%")
                   ->orWhereHas('category', function($qc) use ($q) {
                       $qc->where('name', 'like', "%$q%");
                   });
            });
        }

        if ($from = $request->query('from')) {
            $query->whereDate('date', '>=', $from);
        }
        if ($to = $request->query('to')) {
            $query->whereDate('date', '<=', $to);
        }

        $items = $query->orderBy('date','desc')->orderBy('created_at','desc')->get()->map(function($e){
            return [
                'id'=>$e->id,
                'date'=>$e->date,
                'category'=>$e->category? $e->category->name : '-',
                'title'=>$e->title,
                'description'=>$e->description,
                'amount'=>number_format($e->amount,2),
                'created_at'=>$e->created_at->format('h:i A')
            ];
        });

        return response()->json(['data'=>$items]);
    }

public function dashboard()
{
    $userId = session('user_id');

    $categories = ExpenseCategory::whereNull('user_id')
        ->orWhere('user_id', $userId)
        ->orderByRaw("CASE WHEN user_id IS NULL THEN 0 ELSE 1 END, name")
        ->get();

    return view('user.expenses.dashboard', compact('categories'));
}


public function managedashboard()
{
    abort(404);
}

}
